--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Laboratories
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

require("sim/building.lua")

-- Laboratory class definition, derived from LBuilding
LLaboratory = LBuilding:subclass
{
	__tostring = function(t) return "{Laboratory:" .. tostring(t.name) .. "}" end,
	_Laboratories = {},
}

-------------------------------------------------------------------------------
-- Functions for data description

function Laboratory(t) return LLaboratory:new(t) end

-------------------------------------------------------------------------------
-- "constructor"

function LLaboratory:new(t)
	t = LBuilding.new(self, t)
	table.insert(self._Laboratories, t)
	t.available = 0
	t.stock = { cacao=true,sugar=true }
	return t
end

-------------------------------------------------------------------------------
-- Convenient data access

function LLaboratory:AllLabs()
	return bsgArrayIterator(self._Laboratories)
end

-------------------------------------------------------------------------------
-- Reset

function LLaboratory:ResetOne()
	LBuilding.ResetOne(self)
	self.history = {}
	self.available = 0
	self.stock = { cacao=true,sugar=true }
end

-------------------------------------------------------------------------------
-- Ownership

function LLaboratory:MarkOwned()
	gSim.labOwned = true
	LBuilding.MarkOwned(self)
	if self.port then
		self.port.labOwned = true
	end
end

-------------------------------------------------------------------------------
-- Load and Save

function LLaboratory:SaveOne()
	local t = LBuilding.SaveOne(self)
	if self.owned then
		t = t or {}
		t.available = self.available
		t.stock = self.stock
	end
	return t
end

function LLaboratory:LoadOne(t)
	LBuilding.LoadOne(self, t)
	self.available = t.available or 0
	self.stock = t.stock or { cacao=true,sugar=true }
end

-------------------------------------------------------------------------------
-- Building interaction

function LLaboratory:SelectCharacterFreePlay() return self.character[1] end

function LLaboratory:OnActivate()
	gActiveBuilding = self
	local char,quest = self:SelectCharacter()
	
	if not self.owned then
		-- FIRSTPEEK: LabActivate, time-stamp, weeks, owned, busy, count
		if fpWrite then fpWrite { "LabActivate", gSim.weeks, 'false', 'false', 0 } end

		local price = self.price or 150000
		
		if gSim.rank < 2 then
			DisplayDialog { "ui/chartalk.lua", char=char, body="lab_lowrank", ok="ok" }
		elseif price > gSim.money then
			DisplayDialog { "ui/chartalk.lua", char=char, body="#"..GetString("lab_expensive", bsgDollars(price)), ok="ok" }
		else
			local yn = DisplayDialog { "ui/chartalk.lua", char=char, body="#"..GetString("lab_purchase", bsgDollars(price)), yes="yes",no="no" }
			if yn == "yes" then
				gSim:AdjustMoney(-price)
				self:MarkOwned()
				local msg = GetString("factory_purchased", GetString(self.name), bsgDollars(price))
				gSim:Message(msg)
			end
		end
	end
	if self.owned then
		if self.available > gSim.weeks then
			-- FIRSTPEEK: LabActivate, time-stamp, weeks, owned, busy, count
			if fpWrite then fpWrite { "LabActivate", gSim.weeks, 'true', 'true', 0 } end
		
			DisplayDialog { "ui/chartalk.lua", char=char, body="lab_busy", ok="ok" }
		else
			local c = self:SelectCharacter()
		
			-- See if user has new ingredients and take one of each
			-- Also make sure the lab has at least one ingredient in inventory
			local ok = false
			local newStuff = false
			for ing in LItem:AllIngredients() do
				if self.stock[ing.name] then
					ok = true
				elseif ing.inventory > 0 and ing.type ~= "it_flower" then
					self.stock[ing.name] = true
					ing.inventory = ing.inventory - 1
					newStuff = true
				end
			end
			ok = ok or newStuff
			
			if not ok then
				-- NO ingredients in laboratory stock
				DisplayDialog { "ui/chartalk.lua", char=char, body="lab_noingredients", ok="ok" }

				-- FIRSTPEEK: LabActivate, time-stamp, weeks, owned, busy, count
				if fpWrite then fpWrite { "LabActivate", gSim.weeks, 'true', 'true', 0 } end
			else
				if newStuff then
					-- NEW ingredients taken from inventory and placed in stock
					gSim:InventoryChanged()
					DisplayDialog { "ui/chartalk.lua", char=char, body="lab_newstuff", ok="ok" }
				end

				-- Look for a list of unknown lab recipes possible with the current ingredients and known types
				-- And a list of possible ingredient counts
				local possible = {}
				local possibleCount = 0
				for item in LItem:AllProducts() do
					if item.lab and not item.known and item.type.knownCount > 0 then
						local maybe = true
						for ing,_ in pairs(item.recipe) do
	-- CHECK LAB INVENTORY NOT PLAYER INVENTORY
	--						local i = LItem:ByName(ing)
	--						if i.inventory == 0 then maybe = false break end
							if not self.stock[ing] then maybe = false break end
						end
						if maybe then
							possible[item.name] = item
							possibleCount = possibleCount + 1
						end
					end
				end
				
				-- FIRSTPEEK: LabActivate, time-stamp, weeks, owned, busy, count
				if fpWrite then fpWrite { "LabActivate", gSim.weeks, 'true', 'true', possibleCount } end
				
				-- Activate the lab, let it handle the case where nothing is possible
				-- What are the fewest possible ingredients, and what other possibilities are there?
				local possibleTypes = {}
				local limit = 6
				local type = LProductType:ByName("truffle")
				for _,item in pairs(possible) do
					possibleTypes[item.type.name] = item.type
					if item.recipeSize < limit then
						limit = item.recipeSize
						type = item.type
					end
				end
			
				-- And play the game (with a clean slate)
				self.history = {}
				DisplayDialog { "ui/lab_game.lua", lab=self, char=char, limit=limit, type=type, possibleRecipes=possible, possibleTypes=possibleTypes, possibleCount=possibleCount }
				if gLabDiscovery then DoModal('ui/recipebook.lua') end
				gLabDiscovery = nil
			end
		end
	end
	self:ExitBuilding()
end
